/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright (c) 2015 Adobe Systems Incorporated. All rights reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 */


/*jslint vars: true, plusplus: true, devel: true, nomen: true, indent: 4,
maxerr: 50, node: true, regexp: true */
/*global */
(function () {
    "use strict";
    
    var path            = require('path'),
        EventEmitter    = require("events").EventEmitter,
        watchedPathList = require("./DWWatchedPathList");
    
    /* 
     * @constructor
     * FileWatcher inherits from the EventEmitter class and exports itself
     * as the module.
     */
    var FileWatcher = module.exports = new EventEmitter();
    
    
    /**
     * returns a glob filter spec confirming filter for the speified extension*/
     // say for "txt" files, this will return "**/**.[tT][xX][tT]"
     /* @param {string} extension eg. "txt"
     *  @returns {string} the glob filter for the extension
     */
    var _getGlobExtensionFilter = function (extension) {
        var i = 0,
            len = extension.length,
            filter = "**/**.",
            upperCaseExt = extension.toUpperCase(),
            loweCaseExt = extension.toLowerCase();
        for (i = 0; i < len; i++) {
            filter = filter + "[" + loweCaseExt[i] + upperCaseExt[i] + "]";
        }
        return filter;
    };
    
    /**
     * returns an array of glob filters for the given path accepting only the specified extensions
     * @param {string} path          the base path
     * @param {Array} extensionList an array of extension strings to filter
     * @returns {Array} an arry of glob filters for the path+extensions
     */
    var _getGlobFilter = function (path, extensionList) {
        var i = 0, len = extensionList.length,
            globFilterList = [];
        for (i = 0; i < len; i++) {
            globFilterList.push(path + _getGlobExtensionFilter(extensionList[i]));
        }
        return globFilterList;
    };
    
    /**
     * Watch a file or folder for changes
     * @param path {string} to watch
     * @param {Array} optional extensionList an array of extension strings to watch, other extensions will be ignored
     */
    var watchFileOrFolder = function (path, extensionList) {
        FileWatcher.emit(FileWatcher.FILE_WATCH_NOT_READY, path);
        var watchedPathData = watchedPathList.removeWatchedPath(path);
        if (watchedPathData && watchedPathData.extensionList) {
            extensionList = extensionList || [];
            extensionList.concat(watchedPathData.extensionList);
            if (watchedPathData.watcher) {
                watchedPathData.watcher.close();
            }
        }
        var filteredPathList = extensionList ? _getGlobFilter(path, extensionList) : path;
        var watcherOptions = {
            ignored: /[\/\\]\./,
            useFsEvents: true,
            usePolling: false
        };
        
        var watcher = null;
        if (process.platform === "win32") {
            var CSharpWatcher = require("./CSharpWatcher");
            watcher = new CSharpWatcher.Watcher();
            watcher.watchPath(path, [watcherOptions.ignored], extensionList, FileWatcher);
            FileWatcher.emit(FileWatcher.FILE_WATCH_READY, path);
        } else {
            var watcherReady = false;
            var chokidar = require('chokidar');
            watcher = chokidar.watch(filteredPathList, watcherOptions).on('all', function (event, path, stats) {
                if (watcherReady) {
                    if (event === "change") {
                        FileWatcher.emit(FileWatcher.FILE_CHANGED_EVENT, path, stats);
                    } else if (event === "add") {
                        FileWatcher.emit(FileWatcher.FILE_ADDED_EVENT, path, stats);
                    } else if (event === "unlink") {
                        FileWatcher.emit(FileWatcher.FILE_REMOVED_EVENT, path, stats);
                    }
                }
            });
            watcher.on("ready", function () {
                watcherReady = true;
                FileWatcher.emit(FileWatcher.FILE_WATCH_READY, path);
            });
        }
        watchedPathList.addWatchedPath(path, {"watcher": watcher, "extensionList" : extensionList});
    };
    
    /**
     * unwatch a registered watch path
     * @param {string} path to unwatch
     */
    var unwatchFileOrFolder = function (path) {
        if (watchedPathList.isPathBeingWatched(path)) {
            var watcherData = watchedPathList.removeWatchedPath(path);
            if (watcherData !== true && watcherData && watcherData.watcher) {
                watcherData.watcher.close();
            }
        }
    };
    
    //public API's
    FileWatcher.watchFileOrFolder   = watchFileOrFolder;
    FileWatcher.unwatchFileOrFolder = unwatchFileOrFolder;
    // public events
    FileWatcher.FILE_WATCH_NOT_READY  = "notReady";
    FileWatcher.FILE_WATCH_READY      = "ready";
    FileWatcher.FILE_CHANGED_EVENT    = "change";
    FileWatcher.FILE_ADDED_EVENT      = "add";
    FileWatcher.FILE_REMOVED_EVENT    = "remove";

}());
